/*
 * User GPIB Include File
 */
typedef short word_t;     /* two byte object */


/* fixed_info_struct definition */
struct fixed_info {
    unsigned char	handler_type;
    unsigned char	board_type;
    unsigned char	handler_version[4];
    unsigned int	handler_segment;
    unsigned int	handler_size;
};

/* structure exits one per gpib board in system */
struct board {
        word_t  b_uflags;          /* user flags (see below) */
        char    b_pad,b_sad;       /* primary and secondary gpib address */
        char    b_eos;             /* end-of-string character */
        char    b_ppe;             /* parallel poll enable byte */
	char	b_dma;		   /* 0:programmed I/O; 1:DMA */
        char    b_sig;             /* signal to send */
        word_t  b_tmo;             /* timeout value (see below) */
};

/* board user flags (b_uflags) */
#define EOT     000001  /* assert EOI with last byte of each write */
#define HLD     000002  /* holdoff handshake at end of each read */
#define EOSM    000034  /* end-of-string modes */
#define REOS    000004  /* terminate read on eos */
#define XEOS    000010  /* assert EOI with eos byte */
#define BIN     000020  /* eight-bit compare */
#define TDCL    000040  /* terminate I/O on device clear */
#define TASC    000100  /* terminate I/O on address status change */
#define PP2	000200	/* support local parallel poll */
#define TTICK   000400  /* if set, b_tmo is ticks not scale factor */
#define IST     001000  /* individual status bit */
#define SRE     002000  /* assert REN when SC */
#define SC      004000  /* interface is system controller */
#define TRI     010000  /* use high-speed timing */
#define VSTD    040000  /* use Very Short T1 delay(very high-speed) */
#define NODIS   040000  /* hold SCSI bus during I/O */

struct device {
	word_t d_uflags;
        char d_bna;       /* access board number */
        char d_pad,d_sad; /* primary and secondary gpib address */
        char d_eos;       /* end-of-string character */
        char d_tmo;       /* timeout value */
        char d_ppe;       /* parallel poll enable byte */
};
/* device user flags (d_uflags) */
#if 0 /*defined in b_uflags */
	EOT     000001  /* assert EOI with last byte of each write */
	HLD     000002  /* holdoff handshake at end of each read */
	EOSM    000034  /* end-of-string modes */
	REOS    000004  /*  terminate read on eos */
	XEOS    000010  /*  assert EOI with eos byte */
	BIN     000020  /*  eight-bit compare */
	TDCL    000040  /* terminate I/O on device clear */
	TASC    000100  /* terminate I/O on address status change */
#endif
#define DSPQ    001000  /* save serial poll responses in queue */
#define WRQS    002000  /* wait for RQS before starting I/O */



/* GPIB status bit vector */

#define ERR     (1<<15) /* Error detected */
#define TIMO    (1<<14) /* timeout */
#define END     (1<<13) /* EOI or eos detected */
#define SRQI    (1<<12) /* SRQ detected by CIC */
#define RQS     (1<<11) /* Device requires service */
#define CMPL    (1<<8)  /* DMA completed (SH/AH synch'd) */
#define LOK     (1<<7)  /* local lockout state */
#define REM     (1<<6)  /* remote state */
#define CIC     (1<<5)  /* controller-in-charge */
#define ATN     (1<<4)  /* attention asserted */
#define TACS    (1<<3)  /* talker active */
#define LACS    (1<<2)  /* listener active */
#define DTAS    (1<<1)  /* device trigger state */
#define DCAS    (1<<0)  /* device clear state */

/* GPIB Bus Control Lines bit vector */
#define BUS_DAV         0x100           /* DAV line status bit  */
#define BUS_NDAC        0x200           /* NDAC line status bit */
#define BUS_NRFD        0x400           /* NRFD line status bit */
#define BUS_IFC         0x800           /* IFC line status bit  */
#define BUS_REN         0x1000          /* REN line status bit  */
#define BUS_SRQ         0x2000          /* SRQ line status bit  */
#define BUS_ATN         0x4000          /* ATN line status bits */
#define BUS_EOI         0x8000          /* EOI line status bits */

/* Error messages */

#define EDVR  0  /* System error */
#define ECIC  1  /* Not CIC (or lost CIC during command) */
#define ENOL  2  /* write detected no listeners */
#define EADR  3  /* board not addressed correctly */
#define EARG  4  /* bad argument to function call */
#define ESAC  5  /* function requires board to be SAC */
#define EABO  6  /* asynchronous operation was aborted */
#define ENEB  7  /* non-existent board */
#define EDMA  8  /* DMA hardware error detected */
#define EBTO  9  /* DMA hardware uP bus timeout */
#define EOIP 10  /* new I/O attempted with old I/O in progress  */
#define ECAP 11  /* no capability for intended opeation         */
#define EFSO 12  /* file system operation error                 */
#define EOWN 13  /* shareable board exclusively owned */
#define EBUS 14  /* bus error */
#define ESTB 15  /* serial poll queue overflow */
#define ESRQ 16  /* SRQ line 'stuck' on */
/*488.2*/
#define ETAB 20  /* The return buffer is full */


extern  unsigned   int      ibsta,iberr,ibcnt;

/* Timeout values and meanings                          */
#define TNONE    0   /* Infinite timeout (disabled)     */
#define T10us    1   /* Timeout of 10 us (ideal)        */
#define T30us    2   /* Timeout of 30 us (ideal)        */
#define T100us   3   /* Timeout of 100 us (ideal)       */
#define T300us   4   /* Timeout of 300 us (ideal)       */
#define T1ms     5   /* Timeout of 1 ms (ideal)         */
#define T3ms     6   /* Timeout of 3 ms (ideal)         */
#define T10ms    7   /* Timeout of 10 ms (ideal)        */
#define T30ms    8   /* Timeout of 30 ms (ideal)        */
#define T100ms   9   /* Timeout of 100 ms (ideal)       */
#define T300ms  10   /* Timeout of 300 ms (ideal)       */
#define T1s     11   /* Timeout of 1 s (ideal)          */
#define T3s     12   /* Timeout of 3 s (ideal)          */
#define T10s    13   /* Timeout of 10 s (ideal)         */
#define T30s    14   /* Timeout of 30 s (ideal)         */
#define T100s   15   /* Timeout of 100 s (ideal)        */
#define T300s   16   /* Timeout of 300 s (ideal)        */
#define T1000s  17   /* Timeout of 1000 s (maximum)     */

extern unsigned ibsta,iberr,ibcnt;

/* ioctl codes */
#define IIOC    ('I'<<8)

/* IBGET/IBSET use the form: ioctl(f,IBGET,&board); */
#define IBOLDGET (IIOC|0)   /* get board/device structure */
#define IBOLDSET (IIOC|1)   /* set board/device structure */

#define IBINB    (IIOC|2)       /* special */
#define IBOUTB   (IIOC|3)       /* special */
#define IBDIAG   (IIOC|5) 	/* special */

#define IBGET    (IIOC|6)   /* get board/device structure */
#define IBSET    (IIOC|7)   /* set board/device structure */

/* Secondary address constant used by IBLN() */
#define NO_SAD	       0	/* No secondary address */
#define ALL_SAD	    0xFF	/* send all secondary addresses */

struct ibinfo { 
        unsigned char bcnt,     /* number of boards */
             	      dcnt,     /* number of devcies */
             	      isdev;    /* non-zero if called with device file descriptor */
};
/* IBINFO uses the form: ioctl(f,IBINFO,&ibinfo); */
#define IBINFO   (IIOC|4)       /* get board and device info */

struct ibarg {
        long     buf;           /* buffer address */
        unsigned n;             /* count or argument */
        unsigned ibsta;         /* return status */
        unsigned iberr;         /* return error code */
        unsigned ibcnt;         /* return count */
/*488.2*/
        unsigned short *buf1;          	/* buffer address */
        unsigned short *buf2;          	/* buffer address */
        unsigned short *buf3;          	/* buffer address */
        unsigned short n1;             	/* count or argument */
        unsigned short n2;             	/* count or argument */
        unsigned int lcnt;         	/* long count */
        unsigned char *bufch;        	/* char argument */
};

#define  BusEOI     (unsigned short)0x8000
#define  BusATN     (unsigned short)0x4000
#define  BusSRQ     (unsigned short)0x2000
#define  BusREN     (unsigned short)0x1000
#define  BusIFC     (unsigned short)0x0800
#define  BusNRFD    (unsigned short)0x0400
#define  BusNDAC    (unsigned short)0x0200
#define  BusDAV     (unsigned short)0x0100


/*488.2*/
/*brd and device handles for 488 calls are [BRD|DEV]HANDLE + file descriptor*/
#define BRDHANDLE 32000
#define DEVHANDLE 32256

/*
 *  This macro can be used to easily create an entry in address list
 *  that is required by many of the 488.2 functions.  An address list is
 *  just an array of unsigned integers.  The primary address goes in the
 *  lower 8-bits and the secondary address goes in the upper 8-bits.
 */
#define MakeAddr(pad, sad) ((unsigned short)(((unsigned char)(pad)) | (((unsigned char)(sad))<<8)))

/*
 *  The following two macros are used to "break apart" an address list
 *  entry.  They take an unsigned integer and return either the primary
 *  or secondary address stored in the integer.
 */
#define  GetPAD(val)    ((unsigned char)(val))
#define  GetSAD(val)    ((unsigned char)((val) >> 8))


/*
 *  This value is used to terminate an address list.  It should be
 *  assigned to the last entry.
 */
#define  NOADDR    0xFFFF


/* These are the values that are used by the Send 488.2 command. */
#define  NULLend (unsigned int)0x00  /* Do nothing at the end of a transfer.*/
#define  NLend   (unsigned int)0x01  /* Send NL with EOI after a transfer.  */
#define  DABend  (unsigned int)0x02  /* Send EOI with the last DAB.         */


/* This value is used by the 488.2 Receive command.
 */
#define  STOPend     (unsigned int)0x0100


/* The following constants are used for the second parameter of the
  ibconfig function.  They are the "option" selection codes. */
#define IbcPAD          0x0001  /* Primary Address                            */
#define IbcSAD          0x0002  /* Secondary Address                          */
#define IbcTMO          0x0003  /* Timeout Value                              */
#define IbcEOT          0x0004  /* Send EOI with last data byte?              */
#define IbcPPC          0x0005  /* Parallel Poll Configure                    */
#define IbcREADDR       0x0006  /* Repeat Addressing                          */
#define IbcAUTOPOLL     0x0007  /* Enable/Disable Auto Serial Polling         */
#define IbcCICPROT      0x0008  /* Use the CIC Protocol?                      */
#define IbcIRQ          0x0009  /* Enable/Disable Hardware Interrupts         */
#define IbcSC           0x000A  /* Board is System Controller?                */
#define IbcSRE          0x000B  /* Assert SRE on device calls?                */
#define IbcEOSrd        0x000C  /* Terminate reads on EOS                     */
#define IbcEOSwrt       0x000D  /* Send EOI with EOS character                */
#define IbcEOScmp       0x000E  /* Use 7 or 8-bit EOS compare                 */
#define IbcEOSchar      0x000F  /* The EOS character.                         */
#define IbcPP2          0x0010  /* Use Parallel Poll Mode 2.                  */
#define IbcTIMING       0x0011  /* NORMAL, HIGH, or VERY_HIGH timing.         */
#define IbcDMA          0x0012  /* Use DMA for I/O                            */
#define IbcReadAdjust   0x0013  /* Swap bytes during an ibrd.                 */
#define IbcWriteAdjust  0x0014  /* Swap bytes during an ibwrt.                */
#define IbcEventQueue   0x0015  /* Enable/disable the event queue.            */
#define IbcSPollBit     0x0016  /* Enable/disable the visibility of SPOLL.    */
#define IbcSendLLO      0x0017  /* Enable/disable the sending of LLO.         */
#define IbcSPollTime    0x0018  /* Set the timeout value for serial polls.    */
#define IbcPPollTime    0x0019  /* Set the parallel poll length period.       */
#define IbcNoEndBitOnEOS 0x01A  /* Remove EOS from END bit of IBSTA.          */
#define IbcEndBitIsNormal 0x001a/* Remove EOS from END bit of IBSTA.          */
#define IbcUnAddr       0x001B  /* Enable/disable device unaddressing.        */
#define IbcSignalNumber 0x001C  /* Set the signal to send on ibsgnl events.   */
#define SIGOPTION        0x1c /* used as signal option parameter for IBCONFIG */
#define IbcHSCableLength 0x01F  /* Set the cable length for HS488 transfers.  */
#define IbcLON          0x022 

/* Constants that can be used (in addition to the ibconfig constants)
  when calling the ibask() function.  */

#define IbaPAD          IbcPAD
#define IbaSAD          IbcSAD
#define IbaTMO          IbcTMO
#define IbaEOT          IbcEOT
#define IbaPPC          IbcPPC
#define IbaREADDR       IbcREADDR
#define IbaAUTOPOLL     IbcAUTOPOLL
#define IbaCICPROT      IbcCICPROT
#define IbaIRQ          IbcIRQ
#define IbaSC           IbcSC
#define IbaSRE          IbcSRE
#define IbaEOSrd        IbcEOSrd
#define IbaEOSwrt       IbcEOSwrt
#define IbaEOScmp       IbcEOScmp
#define IbaEOSchar      IbcEOSchar
#define IbaPP2          IbcPP2
#define IbaTIMING       IbcTIMING
#define IbaDMA          IbcDMA
#define IbaReadAdjust   IbcReadAdjust
#define IbaWriteAdjust  IbcWriteAdjust
#define IbaEventQueue   IbcEventQueue
#define IbaSPollBit     IbcSPollBit
#define IbaSendLLO      IbcSendLLO
#define IbaSPollTime    IbcSPollTime
#define IbaPPollTime    IbcPPollTime
#define IbaNoEndBitOnEOS IbcNoEndBitOnEOS
#define IbaEndBitIsNormal IbcEndBitIsNormal
#define IbaUnAddr       IbcUnAddr
#define IbaSignalNumber IbcSignalNumber
#define IbaHSCableLength IbcHSCableLength
#define IbaLON          IbcLON

#define IbaBNA          0x0200  /* A device's access board.                   */
#define IbaBaseAddr     0x0201  /* A GPIB board's base I/O address.           */
#define IbaDmaChannel   0x0202  /* A GPIB board's DMA channel.                */
#define IbaIrqLevel     0x0203  /* A GPIB board's IRQ level.                  */
#define IbaBaud         0x0204  /* Baud rate used to communicate to CT box.   */
#define IbaParity       0x0205  /* Parity setting for CT box.                 */
#define IbaStopBits     0x0206  /* Stop bits used for communicating to CT.    */

#define IbaDataBits     0x0207  /* Data bits used for communicating to CT.    */
#define IbaComPort      0x0208  /* System COM port used for CT box.           */
#define IbaComIrqLevel  0x0209  /* System COM port's interrupt level.         */
#define IbaComPortBase  0x020A  /* System COM port's base I/O address.        */ 


/* the following codes use the form: ioctl(f,IBRD,&ibarg);  */
#define IBTRG   (IIOC|10)
#define IBCLR   (IIOC|11)
#define IBCMD   (IIOC|12)
#define IBXTRC  (IIOC|14)
#define IBRSP   (IIOC|15)
#define IBRPP   (IIOC|16)
#define IBBNA   (IIOC|17)
#define IBWAIT  (IIOC|18)
#define IBONL   (IIOC|19)
#define IBRSV   (IIOC|20)
#define IBGTS   (IIOC|21)
#define IBCAC   (IIOC|22)
#define IBPOKE  (IIOC|23)
#define IBSGNL  (IIOC|24)
#define IBSRE   (IIOC|25)
#define IBRSC   (IIOC|26)
#define IBRD    (IIOC|27)
#define IBWRT   (IIOC|28)
#define IBLOC   (IIOC|29)
#define IBPCT   (IIOC|30)
#define IBSIC   (IIOC|31)
#define IBPPC   (IIOC|33)
#define IBEOT   (IIOC|35)
#define IBPAD   (IIOC|36)
#define IBSAD   (IIOC|37)
#define IBDMA   (IIOC|38)
#define IBEOS   (IIOC|39)
#define IBTMO   (IIOC|40)
#define IBWRTF  (IIOC|41)
#define IBRDF   (IIOC|42)
#define IBLLO   (IIOC|43)
#define IBSTAT  (IIOC|44)
#define IBLN    (IIOC|45)
#define IBDEV   (IIOC|46)
#define IBSIST  (IIOC|47)
#define IBSLINE (IIOC|48)
#define IBCONFIG (IIOC|49)
#define IBASK (IIOC|74)


/* NI-488.2 calls */
#define ALLSPOLL        (IIOC|50)
#define DEVCLEARLIST    (IIOC|51)
#define ENABLELOCAL     (IIOC|52)
#define ENABLEREMOTE    (IIOC|53)
#define FINDRQS         (IIOC|54)
#define FINDLSTN        (IIOC|55)
#define PASSCONTROL     (IIOC|56)
#define PPOLLCONFIG     (IIOC|57)
#define PPOLLUNCONFIG   (IIOC|58)
#define PPOLL           (IIOC|59)
#define RECEIVE         (IIOC|60)
#define RCVRESPMSG      (IIOC|61)
#define RECEIVESETUP    (IIOC|62)
#define RESETSYS        (IIOC|63)
#define SENDCMDS        (IIOC|64)
#define SENDDATABYTES   (IIOC|65)
#define SENDIFC         (IIOC|66)
#define SENDLIST        (IIOC|67)
#define SENDLLO         (IIOC|68)
#define SENDSETUP       (IIOC|69)
#define SETRWLS         (IIOC|70)
#define TESTSYS         (IIOC|71)
#define TRIGGERLIST     (IIOC|72)
#define FINDDEV (IIOC|73)

#define DNMSZ   15


extern void getfd();
#ifdef __STDC__
extern int ibbna(int handle, char *bdname);
extern int ibcac(int handle, int v);
extern int ibclr(int handle);
extern int ibcmd(int handle, char *buffer, long cnt);
extern int ibdev(int boardID, int padval, int sadval, int tmoval, int eotval, int eosval);
extern int ibdma(int handle, int v);
extern int ibeos(int handle, int v);
extern int ibeot(int handle, int v);
extern int ibfind(char *bdname);
extern int ibgts(int handle, int v);
extern int ibist(int handle, int v);
extern int iblines(int handle, short *lines);
extern int ibln(int handle, int padval, int sadval, short *listenflag);
extern int ibloc(int handle);
extern int ibllo(int handle);
extern int ibonl(int handle, int v);
extern int ibpad(int handle, int v);
extern int ibpct(int handle);
extern int ibppc(int handle, int v);
extern int ibrd(int handle, char *buffer, long cnt);
extern int ibrdf(int handle, char *flname);
extern int ibrpp(int handle, char *ppr);
extern int ibrsc(int handle, int v);
extern int ibrsp(int handle, char *spr);
extern int ibrsv(int handle, int v);
extern int ibsad(int handle, int v);
extern int ibsgnl(int handle, int v);
extern int ibsic(int handle, int v);
extern int ibsre(int handle, int v);
extern int ibtmo(int handle, int v);
extern int ibtrg(int handle);
extern int ibwait(int handle, int mask);
extern int ibwrt(int handle, char *buffer, long cnt);
extern int ibwrtf(int handle, char *flname);
extern int ibpoke(int handle, int value);
extern int ibconfig(int handle, int option, int value);
extern int ibask(int handle, int option, int *value);

extern void SendCmds(int boardID, unsigned char *buf, unsigned int cnt);
extern void SendSetup(int boardID, unsigned short listen[]);
extern void SendDataBytes(int boardID, unsigned char *buffer, unsigned int cnt, unsigned short eot_mode);
extern void Send(int boardID, unsigned short listener, unsigned char *databuf, unsigned int datacnt, unsigned short eotMode);
extern void SendList(int boardID, unsigned short listeners[], unsigned char *databuf, unsigned int datacnt, unsigned short eotMode);
extern void ReceiveSetup(int boardID, unsigned short talker);
extern void RcvRespMsg(int boardID, unsigned char *buffer, unsigned int cnt, unsigned short eotMode);
extern void Receive(int boardID, unsigned short talker, unsigned char *buffer, unsigned int cnt, unsigned short eotMode);
extern void SendIFC(int boardID);
extern void DevClear(int boardID, unsigned short address);
extern void DevClearList(int boardID, unsigned short addrlist[]);
extern void EnableLocal(int boardID, unsigned short laddrs[]);
extern void EnableRemote(int boardID, unsigned short laddrs[]);
extern void SetRWLS(int boardID, unsigned short laddrs[]);
extern void SendLLO(int boardID);
extern void PassControl(int boardID, unsigned short talker);
extern void ReadStatusByte(int boardID, unsigned short talker, unsigned short *result);
extern void Trigger(int boardID, unsigned short laddrs);
extern void TriggerList(int boardID, unsigned short laddrs[]);
extern void PPollConfig(int boardID, unsigned short laddr, unsigned short dataLine, unsigned short lineSense);
extern void PPollUnconfig(int boardID, unsigned short laddrs[]);
extern void PPoll(int boardID, unsigned short *res_ptr);
extern void TestSRQ(int boardID, unsigned short *result);
extern void WaitSRQ(int boardID, unsigned short *result);
extern void ResetSys(int boardID, unsigned short laddrs[]);
extern void FindRQS(int boardID, unsigned short taddrs[], unsigned short *dev_stat);
extern void AllSpoll(int boardID, unsigned short taddrs[], unsigned short res[]);
extern void FindLstn(int boardID, unsigned short pads[], unsigned short results[], unsigned short limit);
extern void TestSys(int boardID, unsigned short addrs[], unsigned short result[]);
#else
extern int ibbna();
extern int ibcac();
extern int ibclr();
extern int ibcmd();
extern int ibdev();
extern int ibdma();
extern int ibeos();
extern int ibeot();
extern int ibfind();
extern int ibgts();
extern int ibist();
extern int iblines();
extern int ibln();
extern int ibloc();
extern int ibllo();
extern int ibonl();
extern int ibpad();
extern int ibpct();
extern int ibppc();
extern int ibrd();
extern int ibrdf();
extern int ibrpp();
extern int ibrsc();
extern int ibrsp();
extern int ibrsv();
extern int ibsad();
extern int ibsgnl();
extern int ibsic();
extern int ibsre();
extern int ibtmo();
extern int ibtrg();
extern int ibwait();
extern int ibwrt();
extern int ibwrtf();
extern int ibpoke();
extern int ibconfig();
extern int ibask();


extern void SendCmds();
extern void SendSetup();
extern void SendDataBytes();           
extern void Send(); 
extern void SendList();
extern void ReceiveSetup();
extern void RcvRespMsg();
extern void Receive();     
extern void SendIFC();
extern void DevClear();
extern void DevClearList();
extern void EnableLocal();
extern void EnableRemote();
extern void SetRWLS();
extern void SendLLO();
extern void PassControl();
extern void ReadStatusByte();
extern void Trigger();
extern void TriggerList();
extern void PPollConfig();
extern void PPollUnconfig();
extern void PPoll();
extern void TestSRQ();
extern void WaitSRQ();
extern void ResetSys();
extern void FindRQS();
extern void AllSpoll();
extern void FindLstn();
extern void TestSys();
#endif
